
/*
DIY Autoloader Sketch

Arduino Pro (5v, 16 MHz) w ATmega328

David Merges
*/

//### LIBRARIES ###
  #include "Servo.h"
  #include "AccelStepper.h"

//### LIBRARIES - END ###

//### DEBUG ###
//  #define DEBUG true
  #ifdef DEBUG
    boolean debug_print_once_var = true;
    #define DEBUG_PRINT(x)    Serial.print (x)
    #define DEBUG_PRINTLN(x)  Serial.println (x)
    #define DEBUG_PRINT_ONCE(x) if (debug_print_once_var) Serial.print(x)
    #define DEBUG_PRINTLN_ONCE(x) if (debug_print_once_var) Serial.println(x)              
    #define DEBUG_PRINT_ONCE_STOP debug_print_once_var = false
    #define DEBUG_PRINT_ONCE_RESET debug_print_once_var = true 
  #else
    #define DEBUG_PRINT(x)
    #define DEBUG_PRINTLN(x)
    #define DEBUG_PRINT_ONCE 
    #define DEBUG_PRINTLN_ONCE(x)  
    #define DEBUG_PRINT_ONCE_STOP 
    #define DEBUG_PRINT_ONCE_RESET 
  #endif
//### DEBUG - END ###

//### DEMO ###
//  #define DEMO_MODE true

  #ifdef DEMO_MODE
    #define DEMO  open_odrv()
  #else
    #define DEMO
  #endif
//### DEMO - END ###

// ### OD-MODE ###
  // DEFAULT: const int OD_MODE[]={LOW, HIGH, HIGH, LOW};  
  const int OD_MODE[]={HIGH, LOW, LOW, LOW};
  // {LOW, HIGH,...  od_is_open   <=> (odrv_es_open == LOW  & odrv_es_closed == HIGH)
  // ...HIGH, LOW};  od_is_closed <=> (odrv_es_open == HIGH & odrv_es_closed == LOW)
// ## OD-MODE - END ###

//### DISTANCE, SPEED, ACCELERATION ###
  //CRANE SERVO
    //ANGLES
    const int crane_servo_pos_park = 0;
    const int crane_servo_pos_release = 100;
    //DELAY (servo movement)
    const int crane_servo_fast = 0; 
    const int crane_servo_slow = 10;
  //CRANE
    //DISTANCE
      const int crane_pos_park = 300;
      const int crane_pos_release_odrv = 2000;
      const int crane_pos_odrv_nopickup = 3800;
      const int crane_pos_odrv_movetray = 3800;
      const int crane_pos_release_tray = 5400;
      const int crane_pos_tray_nopickup = 18500;
      
      const int crane_stepstoaddforpickup = 1200;
      const int crane_stepstoaddpickupfail = 1400;
    //SPEED, ACCELERATION
      const int crane_HiSpeed(2000.0);  
      const int crane_LoSpeed(500.0);  
      const int crane_Accel(4000.0);  
      const int crane_delayafterpickup(500);
  //TRAY
    //DISTANCE
      const int tray_pos_pickup = 600;
      const int tray_pos_release = 11100;
    //SPEED, ACCELERATION
      const int tray_HiSpeed(3000.0);  
      const int tray_LoSpeed(500.0);  
      const int tray_Accel(6000.0);  
//### DISTANCE, SPEED, ACCELERATION - END ###

//### PINS ###
  //POWERSUPPLY
    const int atx_ps_on_pin = 12;

    const int mot_enable = 8;
  //HID 
    const int led = 13;
    const int error_led = 9;
    const int start_button = 2;
  //CRANE
    const int crane_disc_present = 3;
    const int crane_mot_dir = 11;
    const int crane_mot_step = 10;
    const int crane_es = 5;
    const int crane_servo_pin = 4;
  //TRAY
    const int tray_es = 7;
    const int tray_mot_dir = A0;
    const int tray_mot_step = A1;
  //ODRV
    const int odrv_button = 6;
    const int odrv_es_closed = A2;
    const int odrv_es_open = A3;
//### PINS - END ###

//### OBJECTS ###
  //create opjects for servocontrol
  Servo crane_servo; 
  //create opjects for steppercontrol
  AccelStepper stepper_crane(AccelStepper::DRIVER, crane_mot_step, crane_mot_dir); 
  AccelStepper stepper_tray(AccelStepper::DRIVER, tray_mot_step, tray_mot_dir); 
//### OBJECTS - END ###

//### VARIABLES ###
  boolean power_state_on;
//### VARIABLES - END ###

void setup()
{
  Serial.begin(9600);
  Serial.write(27);
  Serial.print("[2J"); // clear screen

  DEBUG_PRINTLN("setup()");
  
  //### pinmodes ###
    //POWERSUPPLY
      pinMode(mot_enable, OUTPUT); //(active low)
      //digitalWrite(mot_enable, HIGH);
    //HID
      pinMode(error_led, OUTPUT); 

      pinMode(start_button, OUTPUT); 
      digitalWrite(start_button, HIGH); //(active low)
    //CRANE
      pinMode(crane_disc_present, INPUT);
      digitalWrite(crane_disc_present, HIGH); //(active low)
      pinMode(crane_es, INPUT);
      digitalWrite(crane_es, HIGH); //(active low)
      pinMode(crane_mot_dir, OUTPUT);
      pinMode(crane_mot_step, OUTPUT);
    //TRAY
      pinMode(tray_es, INPUT);
      digitalWrite(tray_es, HIGH); //(active low)
      pinMode(tray_mot_dir, OUTPUT);
      pinMode(tray_mot_step, OUTPUT);
    //ODRV 
      pinMode(odrv_es_closed, INPUT);
      pinMode(odrv_es_open, INPUT);
      pinMode(odrv_button, INPUT); //has to be INPUT for hi-Z (so the button will work)
  //### pinmodes END ###

  //### INITIALISE & HOME ###
    //STEPPER
      stepper_crane.setMaxSpeed(crane_HiSpeed);
      stepper_crane.setAcceleration(crane_Accel);
      stepper_crane.setEnablePin(mot_enable);
      stepper_crane.setPinsInverted(false, false, true);

      stepper_tray.setMaxSpeed(tray_HiSpeed);
      stepper_tray.setAcceleration(tray_Accel);
      stepper_tray.setEnablePin(mot_enable);
      stepper_tray.setPinsInverted(true, false, true); 

      poweron();
  //### INITIALISE & HOME - END ###
}

 

void loop()
{
  DEBUG_PRINTLN("loop()");

  while(!od_is_open() & power_state_on) 
  {
    if(!power_state_on & (start_button == LOW))
      {
        poweron();
      }
  }
  DEBUG_PRINT_ONCE_RESET;
  
  stepper_crane.enableOutputs();
  home_stepper(); 
  
  crane_pickup(crane_pos_odrv_nopickup, crane_stepstoaddforpickup);

  if(digitalRead(crane_disc_present) == HIGH)
  {  
    crane_step_om(crane_pos_park, crane_pos_odrv_nopickup);
  }
  else
  {
    crane_step(crane_pos_park);
  }
    
  close_odrv();

  if(digitalRead(crane_disc_present) == HIGH)
  {
    tray_step(tray_pos_release);
    crane_step(crane_pos_release_tray);
    crane_servo_gotoandpark(crane_servo_pos_release);
    crane_step(crane_pos_odrv_movetray);
    tray_step(tray_pos_pickup);
    crane_pickup(crane_pos_tray_nopickup, crane_stepstoaddforpickup);
    if(digitalRead(crane_disc_present) == HIGH)
    {
      crane_step_om(crane_pos_park, crane_pos_tray_nopickup);
    }
    else
    {
      crane_step(crane_pos_park);
    }
  }
  else
  {
    crane_pickup(crane_pos_tray_nopickup, crane_stepstoaddforpickup);
    if(digitalRead(crane_disc_present) == HIGH)
    {
      crane_step_om(crane_pos_park, crane_pos_tray_nopickup);
    }
    else
    {
      crane_step(crane_pos_park);
    }   
  }

  if(digitalRead(crane_disc_present) == HIGH)
  {
    open_odrv();
    crane_step(crane_pos_release_odrv);
    crane_servo_gotoandpark(crane_servo_pos_release);
    crane_step(crane_pos_park);
    close_odrv();
    stepper_crane.disableOutputs();    
    DEMO;
  }
  else 
  {
    stepper_crane.disableOutputs();    
    atx_ps_on(false);
  }
}



boolean od_is_open()
{
  DEBUG_PRINT_ONCE("od_is_open() - OD_MODE_[");
  DEBUG_PRINT_ONCE(OD_MODE[0]);
  DEBUG_PRINT_ONCE(", ");
  DEBUG_PRINT_ONCE(OD_MODE[1]);
  DEBUG_PRINT_ONCE(", ");
  DEBUG_PRINT_ONCE(OD_MODE[2]);
  DEBUG_PRINT_ONCE(", ");
  DEBUG_PRINT_ONCE(OD_MODE[3]);
  DEBUG_PRINTLN_ONCE("]");
  DEBUG_PRINT_ONCE_STOP;
  return ((digitalRead(odrv_es_open) == OD_MODE[0]) & (digitalRead(odrv_es_closed) == OD_MODE[1]));
}

boolean od_is_closed()
{
  DEBUG_PRINT_ONCE("od_is_closed() - OD_MODE_[");
  DEBUG_PRINT_ONCE(OD_MODE[0]);
  DEBUG_PRINT_ONCE(", ");
  DEBUG_PRINT_ONCE(OD_MODE[1]);
  DEBUG_PRINT_ONCE(", ");
  DEBUG_PRINT_ONCE(OD_MODE[2]);
  DEBUG_PRINT_ONCE(", ");
  DEBUG_PRINT_ONCE(OD_MODE[3]);
  DEBUG_PRINTLN_ONCE("]");
  DEBUG_PRINT_ONCE_STOP;
    return ((digitalRead(odrv_es_open) == OD_MODE[2]) & (digitalRead(odrv_es_closed) == OD_MODE[3]));
}

void atx_ps_on(boolean x)
{
  DEBUG_PRINT("atx_ps_on(");
  DEBUG_PRINT(x);
  DEBUG_PRINTLN(")");
  if (x == true)
  {
    pinMode(atx_ps_on_pin, OUTPUT);  //powersupply power on (active low)
    power_state_on = true;
  }
  else if (x == false)
  {
    pinMode(atx_ps_on_pin, INPUT); //powersupply power off (active low)
    power_state_on = false;
  }
}

void poweron()
{
  DEBUG_PRINTLN("poweron()");
  atx_ps_on(true);
  delay(500);
  stepper_crane.enableOutputs();
  home_crane(); 
  home_tray();
  stepper_crane.disableOutputs();
  crane_servo_goto(crane_servo_pos_park, crane_servo_fast);
}

void open_odrv()
{
  DEBUG_PRINTLN("open_odrv()");
  pinMode(odrv_button, OUTPUT); 
  delay(100);
  pinMode(odrv_button, INPUT);
  while(!od_is_open())
  {
  }  
}

void close_odrv()
{
  DEBUG_PRINTLN("close_odrv()");
  pinMode(odrv_button, OUTPUT); 
  delay(100);
  pinMode(odrv_button, INPUT);
  while(!od_is_closed())
  {
  }  
}

void crane_servo_gotoandpark(int angle)
{
  DEBUG_PRINT("crane_servo_gotoandpark(");
  DEBUG_PRINT(angle);
  DEBUG_PRINTLN(")");
  crane_servo_goto(angle, crane_servo_slow);
  crane_servo_goto(crane_servo_pos_park, crane_servo_fast);
}

void crane_servo_goto(int angle, int inc_delay)
{
  DEBUG_PRINT("crane_servo_goto(");
  DEBUG_PRINT(angle);
  DEBUG_PRINT(", ");
  DEBUG_PRINT(inc_delay);
  DEBUG_PRINTLN(")");

  int cur_angle = crane_servo.read();
  
  if (cur_angle < angle)
  {
    crane_servo.attach(crane_servo_pin);
    for(int i=cur_angle;i < angle; i++)
    { 
      crane_servo.write(i); 
      delay(inc_delay);
    }
    crane_servo.detach();
  }
  else if (cur_angle > angle)
  {
    crane_servo.attach(crane_servo_pin);
    for(int i=cur_angle;i > angle; i--)
    { 
      crane_servo.write(i); 
      delay(inc_delay);
    }
    delay(500);
    crane_servo.detach();
  }
}
  
void home_crane()
{
  DEBUG_PRINTLN("home_crane()");

  stepper_crane.moveTo(-crane_pos_tray_nopickup);
  stepper_crane.setMaxSpeed(crane_LoSpeed);
  
  while(digitalRead(crane_es) == LOW)
  {
    stepper_crane.run();
  }
  stepper_crane.stop();
  stepper_crane.setCurrentPosition(0);
  stepper_crane.setMaxSpeed(crane_HiSpeed);
  stepper_crane.moveTo(crane_pos_park);
  while(stepper_crane.distanceToGo() != 0)
  {
    stepper_crane.run();
  }
}

void home_tray()
{
  DEBUG_PRINTLN("home_tray()");
  
  stepper_tray.moveTo(-crane_pos_tray_nopickup);
  stepper_tray.setMaxSpeed(crane_LoSpeed);
  
  while(digitalRead(tray_es) == LOW)
  {
    stepper_tray.run();
  }
    stepper_tray.stop();
    stepper_tray.setCurrentPosition(0);
    stepper_tray.setMaxSpeed(crane_HiSpeed);
    
    stepper_tray.moveTo(crane_pos_park);
    while(stepper_tray.distanceToGo() != 0)
    {
    stepper_tray.run();
    }
}

void home_stepper()
{
  DEBUG_PRINTLN("home_stepper()");
  
  boolean homing_crane = true;
  boolean homing_tray = true;

  stepper_crane.setMaxSpeed(crane_LoSpeed);
  stepper_tray.setMaxSpeed(tray_LoSpeed);

  if ((digitalRead(crane_es) == HIGH) or (digitalRead(tray_es) == HIGH)) 
  {
    DEBUG_PRINTLN("es_error");
    while(1) //Error handling needs to be done ???
    {
    }
  }
  else
  {
    stepper_crane.moveTo(stepper_crane.currentPosition()-crane_Accel);
    stepper_tray.moveTo(stepper_tray.currentPosition()-tray_Accel);
    
    while ((stepper_crane.distanceToGo() != 0) or (stepper_tray.distanceToGo() != 0))
    {
      stepper_crane.run();
      stepper_tray.run();
      
      if (homing_crane)
      {
        if (digitalRead(crane_es) == HIGH)
        {
          stepper_crane.stop();
          stepper_crane.setCurrentPosition(0);

          stepper_crane.setMaxSpeed(crane_HiSpeed);
          stepper_crane.moveTo(crane_pos_park);
          homing_crane = false;
        }
      }

      if (homing_tray)
      {
        if (digitalRead(tray_es) == HIGH)
        {
          stepper_tray.stop();
          stepper_tray.setCurrentPosition(0);

          stepper_tray.setMaxSpeed(tray_HiSpeed);
          stepper_tray.moveTo(tray_pos_pickup);
          homing_tray = false;
        }
      }
    }
  }
}

void crane_pickup(int pos, int addstepps)
{
  int startpos = stepper_crane.currentPosition();
  boolean doitonce = false;

  DEBUG_PRINT("crane_pickup(");
  DEBUG_PRINT(pos);
  DEBUG_PRINT(", ");
  DEBUG_PRINT(addstepps);
  DEBUG_PRINT(") - startpos: ");
  DEBUG_PRINTLN(startpos);

  if (startpos > pos)
  {
    DEBUG_PRINTLN("pickup error");
  }
  if (startpos <= pos)
  {
    stepper_crane.moveTo(pos);
    
    while(stepper_crane.distanceToGo() != 0)
    {
      stepper_crane.run();
      
      if (digitalRead(crane_disc_present) == HIGH)
      {
        if(doitonce == false)
        {
          DEBUG_PRINT("crane_pickup at: ");
          DEBUG_PRINTLN(stepper_crane.currentPosition());
          stepper_crane.moveTo((stepper_crane.currentPosition() + addstepps));
          doitonce = true;
        }
      }
    }
    delay(crane_delayafterpickup);
  }  
}

void crane_step_om(int pos, int pospickup)
{
  DEBUG_PRINT("crane_step_om(");
  DEBUG_PRINT(pos);
  DEBUG_PRINT(", ");
  DEBUG_PRINT(pospickup);
  DEBUG_PRINTLN(")");

  int i = 0;
  int e1 = 3;
  int e2 = 6;

  stepper_crane.moveTo(pos);

  while(stepper_crane.distanceToGo() != 0)
  {
    stepper_crane.run();
    if (digitalRead(crane_disc_present) == LOW)
    {
      i++;
      DEBUG_PRINT("crane_step_om_error: ");
      DEBUG_PRINT(i);
      DEBUG_PRINT(" at: ");
      DEBUG_PRINTLN(stepper_crane.currentPosition());
      if (i <= e1)
      {
        crane_pickup(pospickup, crane_stepstoaddforpickup);
        stepper_crane.moveTo(pos);
      }
      else if (i <= e2)
      {
        crane_pickup(pospickup, crane_stepstoaddpickupfail);
        stepper_crane.moveTo(crane_pos_release_odrv);
      }
      else
      {
        crit_error();
      }
    }            
  }    
  if (i >= e1)
  {
    home_crane();
    crane_step(pos);
  }
}

void crit_error()
{
  DEBUG_PRINTLN("crit_error())");

  digitalWrite(error_led, HIGH);
  atx_ps_on(false);
  while (1)
  {}
}

void crane_step(int pos)
{
  DEBUG_PRINT("crane_step(");
  DEBUG_PRINT(pos);
  DEBUG_PRINTLN(")");
  stepper_crane.moveTo(pos);
    
  while(stepper_crane.distanceToGo() != 0)
  {
    stepper_crane.run();
  }
}

void tray_step(int pos)
{
  DEBUG_PRINT("tray_step(");
  DEBUG_PRINT(pos);
  DEBUG_PRINTLN(")");
  stepper_tray.moveTo(pos);
  stepper_tray.setMaxSpeed(tray_HiSpeed);
  
  while(stepper_tray.distanceToGo() != 0)
  {
    stepper_tray.run();
  }
}
